﻿using log4net;
using Microsoft.Xrm.Sdk;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Net.Http;
using System.Net.Http.Headers;
using System.Threading.Tasks;
using VA.PPMS.Context;
using VA.PPMS.Context.Interface;
using VA.PPMS.IWS.BatchService.Interface;
using VA.PPMS.IWS.BlobService.Interface;
using VA.PPMS.IWS.Common;
using VA.PPMS.IWS.Functions.Configuration.Interface;
using VA.PPMS.IWS.MappingService.Interface;
using VA.PPMS.IWS.PersistenceService.Interface;
using VA.PPMS.IWS.ProviderService.Interface;
using VA.PPMS.IWS.SchemaValidationService.Interface;
using VA.PPMS.ProviderData;

namespace VA.PPMS.IWS.ProviderService
{
    public class ProviderService : IProviderService
    {
        private readonly ILog _logger;
        private readonly IIwsConfiguration _configuration;
        private readonly IBatchService _batchService;
        private readonly IBlobService _blobService;
        private readonly ISchemaValidationService _schemaValidationService;
        private readonly IMappingService _mappingService;
        private readonly IPersistenceService _persistenceService;
        private readonly IPpmsHelper _ppmsHelper;

        public ProviderService(ILog logger, IIwsConfiguration configuration, IBatchService batchService, IBlobService blobService, ISchemaValidationService schemaValidationService, IMappingService mappingService, IPersistenceService persistenceService, IPpmsHelper ppmsHelper)
        {
            _logger = logger;
            _configuration = configuration;
            _batchService = batchService;
            _blobService = blobService;
            _schemaValidationService = schemaValidationService;
            _mappingService = mappingService;
            _persistenceService = persistenceService;
            _ppmsHelper = ppmsHelper;
        }

        public async Task ProcessProvider(string message)
        {
            try
            {
                var dasMessage = new DasMessage(message);

                if (!dasMessage.IsValid) throw new PpmsServiceException("CCN Data Request - Unable to process request, data is invalid");

                var packet = await _configuration.GetResponseDocumentReferenceUrlAsync();
                dasMessage.Content = $"{packet}{dasMessage.Content}"; // string.Format(packet.UrlPattern, packet.Host, packet.Port, dasMessage.Content);

                if (await CallUrl(dasMessage))
                {
                    await _batchService.UpdateBatch(dasMessage, "CCN Initial Data Request processed.", (int)ppms_batch_StatusCode.DataRequested);
                }
                else
                {
                    await _batchService.UpdateBatch(dasMessage, "CCN Initial Data Request failed GET request.", (int)ppms_batch_StatusCode.Failed);
                }
            }
            catch (Exception ex)
            {
                _logger.Error($"@@@@ ERROR - There was a problem processing Provider. @@@@", ex);
                throw;
            }
        }

        public async Task ProcessProviderPayload(string dasMessage)
        {
            try
            {
                var message = new DasMessage(dasMessage);
                var transactionId = message.TransactionId;
                var timer = new Stopwatch();

                _logger.Info($">>> INFO - Start Orchestration Processing <<<");
                timer.Start();

                //var batch = _ppmsHelper.GetBatch(message);
                //if (batch == null)
                //{
                //    _logger.Info($"---- INFO - Create Batch ----");
                //    await _ppmsHelper.CreateBatch(message, "Submission of data directly.", (int)ppms_batch_StatusCode.DataRequestValidated);
                //}

                var providers = await _blobService.GetBlobAsync(dasMessage);

                if (providers == null)
                {
                    _logger.Error($"@@@@ ERROR - Unable to get blob.");
                    return;
                }

                _logger.Info($"---- INFO - Validate schema ----");
                providers.IsVaNetwork = message.IsVaNetwork;
                var validationResult = await _schemaValidationService.ValidateSchemaAsync(providers);

                var validationSuccesses = validationResult.Where(x => string.IsNullOrEmpty(x.Result)).ToList();
                var result = providers.Provider.Join(validationSuccesses,
                        provider => provider.ProviderId,
                        successes => successes.ProviderId,
                        (provider, successes) => provider)
                    .ToList();

                var newProviders = new Providers
                {
                    NetworkId = message.SenderId,
                    TransactionId = message.TransactionId,
                    Provider = result,
                    IsVaNetwork = providers.IsVaNetwork
                };

                var batchList = new List<Validation>();
                var mapResults = new MapperResult();

                if (newProviders.Provider.Any())
                {
                    _logger.Info($"---- INFO - Begin Processing Providers ({newProviders.Provider.Count}) ----");
                    var size = newProviders.Provider.Count;
                    MapperResult results = null;

                    // Retrieve default owner
                    _logger.Info("---- INFO - Retrieve CCN Owner record. ----");
                    EntityReference owner = null;
                    if (!newProviders.IsVaNetwork) owner = await _ppmsHelper.GetCcnOwner();

                    _logger.Info($"---- INFO - Mapping provider info... ----");
                    foreach (var item in newProviders.Provider)
                    {
                        try
                        {
                            var list = new Providers
                            {
                                TransactionId = message.TransactionId,
                                NetworkId = message.SenderId,
                                Provider = new List<Provider> { item },
                                IsVaNetwork = newProviders.IsVaNetwork
                            };

                            if (owner != null) list.OwnerId = owner.Id;

                            results = await _mappingService.MapAsync(list);
                        }
                        catch (Exception ex)
                        {
                            var msg = $"An error occurred trying to map provider info. Error: { ex.Message}";
                            if (results == null)
                            {
                                results = new MapperResult(msg);
                            }
                            else
                            {
                                results.Details[0].ValidationMessage = msg;
                            }

                            _logger.Error($"@@@@ ERROR - Unable to map provider info. @@@@");
                        }

                        if (results != null)
                        {
                            mapResults.Add(results.Details);
                        }
                    }
                }

                _logger.Info($"---- INFO - Saving validation results ----");

                UpdateValidations(validationResult, mapResults.Details);

                foreach (var validation in validationResult)
                {
                    try
                    {
                        var account = mapResults.FindProvider(validation.ProviderId);
                        if (account != null && string.IsNullOrEmpty(validation.CorrelationId))
                        {
                            validation.CorrelationId = account.Id.ToString();
                        }
                        batchList.Add(validation);
                    }
                    catch (Exception)
                    {
                        _logger.Error($"@@@@ ERROR - Unable to save batch info. @@@@");
                    }
                }

                if (batchList.Any())
                {
                    _logger.Info($"---- INFO - Saving batch info... ----");
                    await _persistenceService.SaveBatchAsync(batchList, message);
                }

                timer.Stop();
                _logger.Info($"---- INFO - Elapsed Time {timer.Elapsed} ----");
                _logger.Info($">>> INFO - End Orchestration Processing <<<");

                _logger.Info($"End ProviderPayloadQueueTrigger function processed");
            }
            catch (Exception e)
            {
                var message = dasMessage ?? "Undefined";
                _logger.Error($"!!!! ERROR - There was a problem with Orchestration Processing. !!!!");
                _logger.Error($"{e.Message}");
            }
        }

        private static async Task<bool> CallUrl(DasMessage message)
        {
            var url = message.Content;
            if (string.IsNullOrEmpty(url)) throw new PpmsServiceException("CCN Data Request - Unable to process request, URL is invalid");

            using (var client = new HttpClient())
            {
                client.BaseAddress = new Uri(url);
                client.DefaultRequestHeaders.Accept.Clear();
                client.DefaultRequestHeaders.Accept.Add(new MediaTypeWithQualityHeaderValue("text/plain"));
                client.DefaultRequestHeaders.Add("X-ConversationID", message.ConversationId);
                client.DefaultRequestHeaders.Add("X-TransactionID", message.TransactionId);
                // Reverse sender and receiver
                client.DefaultRequestHeaders.Add("X-RoutingSenderID", message.ReceiverId);
                client.DefaultRequestHeaders.Add("X-RoutingReceiverIDs", message.SenderId);
 
                var response = await client.GetAsync(string.Empty);

                // TODO - Complete the implementation. Not sure what to do with the result
                return response.IsSuccessStatusCode;
            }
        }

        private static void UpdateValidations(List<Validation> validations, IList<MapperResultDetail> details)
        {
            foreach (var detail in details)
            {
                if (detail.IsValid) continue;

                var validation = validations.FirstOrDefault(x => x.ProviderId == detail.ProviderId);
                if (validation != null)
                {
                    validation.Result = detail.ValidationMessage;
                }
            }
        }
    }
}